/*
Copyright (C) 2010 Edge-Core Networks
This software file (the "File") is owned and distributed by 
Edge-Core Networks under the following licensing terms.

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License as
published by the Free Software Foundation; either version 2 of
the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston,
MA 02111-1307 USA
*/

/* Module Name: AMS_PART.C
 * Purpose:
 *      This module Support to get partition information by index/type/name
 *
 * Notes:
 *
 * Copyright(C)  Edge-Core Networks , 2010
 * Autor: alvin ling 
 * jerry.du rewrite for filemapping management 20080630
 */

#include <common.h>
#include <asm/byteorder.h> /* for ntohl() */
#include <command.h>

/*#ifdef CONFIG_AMS_PARTITION*/

#include "ams_common.h"
#include "ams_part.h"

/*
 * you can change AMS_PART_LOG_LEVEL to 4 or smaller to enable debug msg in uc_mgr.c
 * you should use AMS_PRINTK("<%d>...", AMS_PART_LOG_LEVEL, ...) 
 * if you are adding debug msg 
 */
#define AMS_PART_LOG_LEVEL 5


#ifdef _USR_DEBUG // ,  debug print
#undef debug
#define debug(fmt,args...)	printf (fmt ,##args)
#endif	/* _USR_DEBUG */

/*
  * Local Variable Declare
  */
static AMS_PARTITION_TABLE_T *pt_ptr = (AMS_PARTITION_TABLE_T *)PARTITION_MANAGEMENT_BASE_ADDR;
static AMS_FILEMAPPING_MGR_T *filemapping_mgr_ptr;
static AMS_FILEMAPPING_MGR_T current_filemapping_mgr;

#ifdef CONFIG_AMS_COMPACT_FLASH
extern unsigned long AMS_CFLIB_read_data(unsigned long sector,unsigned long sectors, unsigned char *buf);
extern unsigned long AMS_CFLIB_write_data(unsigned long sector,unsigned long sectors, unsigned char *buf, unsigned char format);
#endif

extern void memmovel(unsigned long dest, unsigned long src, size_t count);
extern int do_delete (cmd_tbl_t *cmdtp, int flag, int argc, char *argv[]);
extern unsigned long AMS_CMD_PART_set_startup_flag(unsigned long index, unsigned char flag);

static UI8_T _str[256]={0};
#define GET_STRING(str) (char*)memcpy(memset(_str,0,sizeof(_str)),(str),sizeof(str))
#define SET_STRING(str,value) (char*)memcpy(memset((str),0,sizeof(str)),(value),(strlen(value)<sizeof(str))?strlen(value):sizeof(str))
#define SET_NSTRING(str,value,_size)  (char*)memcpy(memset((str),0,sizeof(str)),(value),(strlen(value)<_size)?strlen(value):_size)

/*-------------------------------------------------------------------------
 * FUNCTION NAME - CheckSum
 * ------------------------------------------------------------------------
 * PURPOSE  :   This funtion will caculate the checksum of the data
 * INPUT      :    start_addr - start address of the buffer
 *                      length        - the length of the data
 * OUTPUT   :    None
 * RETURN    :   The checksum of the data
 * NOTES     :   None
 * ------------------------------------------------------------------------
 */
unsigned long CheckSum(unsigned long start_addr, unsigned long length)
{
	unsigned long checksum, mask;
	unsigned long i;
	
	checksum = 0; 
	
	/* loop 32-bit a time */
	for (i = 0; i < length; i += 4)
	{
		if (i+4 > length)
			mask = 0xffffffff << ((i+4-length)<<3);
		else
			mask = 0xffffffff;

		checksum ^= (*(unsigned long*)(start_addr+i) & mask);
	}
	
	return checksum;
}

static unsigned long
AMS_PART_check_type(unsigned long index, AMS_PARTITION_TYPE_T type)
{
    if (index > MAX_PARTITION_NUMBER)
    {
        AMS_PRINTK("<%d>%s:invalid partition index\n", AMS_PART_LOG_LEVEL, __FUNCTION__);
        return RC_PARAMENT_ERROR;
    }

    if (pt_ptr->flash_region[index].region_type == type)
        return RC_OK;

    return RC_ERROR;
}

static unsigned long
AMS_PART_check_free(unsigned long index)
{
    AMS_PART_get_filemapping_mgr_ptr(&filemapping_mgr_ptr);
    
    if (index > MAX_PARTITION_NUMBER)
    {
        AMS_PRINTK("<%d>%s:invalid partition index\n", AMS_PART_LOG_LEVEL, __FUNCTION__);
        return RC_PARAMENT_ERROR;
    }

#ifdef CONFIG_ACCTON_MULTI
    if (pt_ptr->flash_region[index].region_type == TYPE_RUNTIME)
    {
        if (filemapping_mgr_ptr->filemapping[index].startup_flag & AMS_PARTITION_FREE)
            return RC_OK;
        return RC_ERROR;
    }
#endif
#ifdef CONFIG_ACCTON_KERNEL_ROOTFS
    if (pt_ptr->flash_region[index].region_type == TYPE_KERNEL  || 
	    pt_ptr->flash_region[index].region_type == TYPE_ROOT_FS)
    {
	    if (filemapping_mgr_ptr->filemapping[index].startup_flag & AMS_PARTITION_FREE)
		 	return RC_OK;
	    return RC_ERROR;
    }
#endif
	return RC_OK;
}

/*-------------------------------------------------------------------------
 * FUNCTION NAME - AMS_PART_check_active
 * ------------------------------------------------------------------------
 * PURPOSE  :   This funtion will check partition active mode.
 * INPUT    :   None
 * OUTPUT   :   None
 * RETURN   :   
 *              RC_OK  =  active mode
 *              RC_ERROR = inactive mode
 * NOTES    :   None
 * ------------------------------------------------------------------------
 */
unsigned long
AMS_PART_check_active(unsigned long index, AMS_PARTITION_TYPE_T type)
{
    AMS_PART_get_filemapping_mgr_ptr(&filemapping_mgr_ptr);
    
    if (index > MAX_PARTITION_NUMBER)
    {
        AMS_PRINTK("<%d>%s:invalid partition index\n", AMS_PART_LOG_LEVEL, __FUNCTION__);
        return RC_PARAMENT_ERROR;
    }

    if (filemapping_mgr_ptr->filemapping[index].startup_flag & AMS_PARTITION_ACTIVE)
	 	return RC_OK;

    return RC_ERROR;
}

/*-------------------------------------------------------------------------
 * FUNCTION NAME - AMS_PART_check_partition
 * ------------------------------------------------------------------------
 * PURPOSE  :   This funtion will check partition validity
 * INPUT      :   None
 * OUTPUT   :   None
 * RETURN    :   0/-1
 * NOTES     :   None
 * ------------------------------------------------------------------------
 */
unsigned long 
AMS_PART_check_partition(void)
{
    if(pt_ptr->magic != PARTITION_TABLE_MAGIC)
        return RC_ERROR;

    if(pt_ptr->header_checksum != CheckSum((unsigned long)&pt_ptr->magic, 16))
        return RC_ERROR;

    return RC_OK;
}

/*-------------------------------------------------------------------------
 * FUNCTION NAME - AMS_PART_get_device_type
 * ------------------------------------------------------------------------
 * PURPOSE  :   This funtion will get device type
 * INPUT      :   device_name = the device name
 * OUTPUT   :   device_type = the device type
 * RETURN    :   -1 not found 0 success
 * NOTES     :   None
 * ------------------------------------------------------------------------
 */
unsigned long 
AMS_PART_get_device_type(const char *device_name, AMS_PARTITION_DEVICE_TYPE_T *device_type)
{
    if (device_name == NULL || device_type == NULL)
    {
        AMS_PRINTK("<%d>%s:invalid params\n", AMS_PART_LOG_LEVEL, __FUNCTION__);
        return RC_PARAMENT_ERROR;
    }
	
    if (strcmp(device_name, DEVICE_NOR_NAME) == 0)
    {
        *device_type = TYPE_DEVICE_NOR;
        return RC_OK;
    }
	  
    if (strcmp(device_name, DEVICE_NAND_NAME) == 0)
    {
        *device_type = TYPE_DEVICE_NAND;
        return RC_OK;
    }
	  
    if (strcmp(device_name, DEVICE_CF_NAME) == 0)
    {
        *device_type = TYPE_DEVICE_CFCARD;
        return RC_OK;
    }

    if (strcmp(device_name, DEVICE_SPI_NAME) == 0)
    {
        *device_type = TYPE_DEVICE_SPI;
        return RC_OK;
    }

    return RC_ERROR;
}

/*-------------------------------------------------------------------------
 * FUNCTION NAME - AMS_PART_get_filemapping
 * ------------------------------------------------------------------------
 * PURPOSE  :   This funtion will get partition region from partition table
 * INPUT      :   index = the index of partition
 * OUTPUT   :   region = the region of the given partition index. 
 * RETURN    :   The execute of this function
 * NOTES     :   None
 * ------------------------------------------------------------------------
 */
unsigned long 
AMS_PART_get_filemapping(unsigned long index, AMS_FILEMAPPING_T *filemapping_ptr)
{
    if (filemapping_ptr == NULL)
    {
        AMS_PRINTK("<%d>%s:invalid params\n", AMS_PART_LOG_LEVEL, __FUNCTION__);
        return RC_PARAMENT_ERROR;
    }
    
    if (AMS_PART_check_partition() != 0)
        return RC_PART_CHECK_ERROR;

    if (index < 0 || index >= MAX_PARTITION_NUMBER)
        return RC_PART_NUMBER_EXCEED;

    if (AMS_PART_get_filemapping_mgr_ptr(&filemapping_mgr_ptr) != RC_OK)
        return RC_ERROR;
    *filemapping_ptr = filemapping_mgr_ptr->filemapping[index];

    return RC_OK;
}

unsigned long 
AMS_PART_get_filemapping_region(FlashRegion_T *region_ptr)
{
    int index;
    FlashRegion_T region;

    if (region_ptr == NULL)
    {
        AMS_PRINTK("<%d>%s:invalid params\n", AMS_PART_LOG_LEVEL, __FUNCTION__);
        return RC_PARAMENT_ERROR;
    }

    for (index = 0; index < MAX_PARTITION_NUMBER; index++)
    {
        memset(&region, 0, sizeof(FlashRegion_T));
        if (AMS_PART_get_region(index, &region) == 0)
            if (region.region_type == TYPE_FILEMAPPING_TABLE)
                break;
    }

    if (index == MAX_PARTITION_NUMBER)
        return RC_ERROR;

    //printf("AMS_PART_get_filemapping_region 0x%08x\n", region.region_base);
    *region_ptr = region;

    return RC_OK;
}

unsigned long 
AMS_PART_init_filemapping(unsigned long index, AMS_FILEMAPPING_T *filemapping_ptr)
{
    if (filemapping_ptr == NULL)
    {
        AMS_PRINTK("<%d>%s:invalid params\n", AMS_PART_LOG_LEVEL, __FUNCTION__);
        return RC_PARAMENT_ERROR;
    }
	
    filemapping_ptr->file_name[0] = '\0';
    filemapping_ptr->file_size = 0;
    filemapping_ptr->creat_time = 0;
    //filemapping_ptr->part_index = index;
    filemapping_ptr->file_type = TYPE_NOT_USED;
    //filemapping_ptr->part_type = PART_TYPE_NO_USED;
    filemapping_ptr->startup_flag = AMS_PARTITION_FREE;
    //memset((char *)filemapping_ptr->reserved, 0, sizeof(unsigned long) * 40);
    //filemapping_ptr->part_checksum =  CheckSum((unsigned long)&filemapping_ptr->file_name[0], sizeof(AMS_FILEMAPPING_T) - sizeof(unsigned long));

    return RC_OK;
}

/*-------------------------------------------------------------------------
 * FUNCTION NAME - AMS_PART_get_number
 * ------------------------------------------------------------------------
 * PURPOSE :   This funtion will the total number of the partition table
 * INPUT      :   None
 * OUTPUT   :   num - the number of the partition table
 * RETURN   :   0
 * NOTES     :   NONE
 * ------------------------------------------------------------------------
 */
static unsigned long
AMS_PART_get_number(unsigned long *num)
{
    if (num == NULL)
    {
        AMS_PRINTK("<%d>%s:invalid params\n", AMS_PART_LOG_LEVEL, __FUNCTION__);
        return RC_PARAMENT_ERROR;
    }

    *num = MAX_PARTITION_NUMBER;

    return RC_OK;
}

unsigned long
AMS_PART_get_partition_table_ptr(AMS_PARTITION_TABLE_T **ptr)
{
    *ptr= pt_ptr;
    return RC_OK;
}

unsigned long
AMS_PART_get_filemapping_mgr_ptr(AMS_FILEMAPPING_MGR_T **ptr)
{
    FlashRegion_T region;
    
    if (AMS_PART_get_filemapping_region(&region) != RC_OK)
        return RC_ERROR;
    *ptr= (AMS_FILEMAPPING_MGR_T *)region.region_base;

    return RC_OK;
}

/*-------------------------------------------------------------------------
 * FUNCTION NAME - AMS_PART_get_startup_by_type
 * ------------------------------------------------------------------------
 * PURPOSE  :   This funtion will get startup-flag partition index from filemapping table
 * INPUT      :   type = partition type
 * OUTPUT   :   partiton_idx = partition index
 * RETURN    :   The result of the function.
 * NOTES     :   NONE
 * ------------------------------------------------------------------------
 */
unsigned long 
AMS_PART_get_startup_by_type(AMS_PARTITION_TYPE_T type,  unsigned long *partiton_idx)
{
    int i;
    
    if (partiton_idx == NULL)
    {
        AMS_PRINTK("<%d>%s:invalid params\n", AMS_PART_LOG_LEVEL, __FUNCTION__);
        return RC_PARAMENT_ERROR;
    }

    AMS_PART_get_filemapping_mgr_ptr(&filemapping_mgr_ptr);
    for (i = 0; i < MAX_PARTITION_NUMBER; i++)
    {	
        if ((pt_ptr->flash_region[i].region_type == type) &&
            (filemapping_mgr_ptr->filemapping[i].startup_flag & AMS_PARTITION_ACTIVE))
        {
            *partiton_idx = i;
            return RC_OK;
        }
    }

    return RC_ERROR;
}

/*-------------------------------------------------------------------------
 * FUNCTION NAME - AMS_PART_get_info_by_index
 * ------------------------------------------------------------------------
 * PURPOSE  :   This funtion will get partition address & size from partition table
 * INPUT      :   index = partition index
 * OUTPUT   :   addr = the address of the partition
                      size  = the size of partition 
 * RETURN    :   The result of the function.
 * NOTES     :   NONE
 * ------------------------------------------------------------------------
 */
unsigned long 
AMS_PART_get_info_by_index(unsigned long index, unsigned long *addr, unsigned long *size)
{
    FlashRegion_T region;

    if (addr == NULL || size == NULL)
    {
        AMS_PRINTK("<%d>%s:invalid params\n", AMS_PART_LOG_LEVEL, __FUNCTION__);
        return RC_PARAMENT_ERROR;
    }

    if (AMS_PART_get_region(index, &region) == 0)
    {
        if ((region.region_type == TYPE_NOT_USED) || (region.region_type == TYPE_MAX))
            return RC_ERROR;
        *addr = region.region_base;
        *size = region.region_size;
        return RC_OK;
    }
    else
        return RC_ERROR;
}

/*-------------------------------------------------------------------------
 * FUNCTION NAME - AMS_PART_get_info_by_type
 * ------------------------------------------------------------------------
 * PURPOSE  :   This funtion will get partition address & size from partition table
 * INPUT      :   type = partition type
 * OUTPUT   :   addr = the address of the partition
                      size  = the size of partition 
 * RETURN    :   0/-1
 * NOTES     :   If there are more than one partition with same type, the function will 
 *                   search the active partition first, if there is no ative partition, it will get
 *                   the first paritition that match the given type.
 * ------------------------------------------------------------------------
 */
unsigned long 
AMS_PART_get_info_by_type(AMS_PARTITION_TYPE_T type, 
                          unsigned long *addr, 
                          unsigned long *size,
                          unsigned long part_index,
                          AMS_PARTITION_DEVICE_TYPE_T *device_type)
{
    unsigned char part_exist = FALSE;
    unsigned char part_free = FALSE;
    FlashRegion_T region;

    if (addr == NULL || size == NULL || device_type == NULL)
    {
        AMS_PRINTK("<%d>%s:invalid params\n", AMS_PART_LOG_LEVEL, __FUNCTION__);
        return RC_PARAMENT_ERROR;
    }
    
    if (AMS_PART_check_type(part_index, type) == 0)
    {
    	if (AMS_PART_check_free(part_index) == 0 &&  AMS_PART_check_active(part_index, type) == RC_ERROR)
    	{
            if (AMS_PART_get_region(part_index, &region) == 0)
            {
                *addr = region.region_base;
                *size = region.region_size;
                if (AMS_PART_get_device_type((char *)region.device_name, device_type) != 0)
                {
                    AMS_PRINTK("<%d>%s:get device type error\n", AMS_PART_LOG_LEVEL, __FUNCTION__);
                    return RC_ERROR;
                }
                return RC_OK;
            }
            part_free = TRUE;
        }
        part_exist = TRUE;
    }

    if (!part_exist)
    {
        printf("Partition does not exist\n");
        return RC_ERROR;
    }

    if (!part_free)
    {
        if(AMS_PART_check_active(part_index, type) == RC_OK){
            printf("This partition is active mode.\n");            
        }else{
            printf("There is no free region.\n");
        }
        return RC_ERROR;
    }

    return RC_OK;
}

unsigned long 
AMS_PART_set_part_info_by_index(unsigned long part_index, const char *name, 
                                unsigned long file_size,
                                AMS_PARTITION_TYPE_T type)
{
    if (name == NULL)
    {
        AMS_PRINTK("<%d>%s:invalid name\n", AMS_PART_LOG_LEVEL, __FUNCTION__);
        return RC_ERROR;
    }

	if (part_index < 0 || part_index > MAX_PARTITION_NUMBER)
    {
        AMS_PRINTK("<%d>%s:invalid partition index\n", AMS_PART_LOG_LEVEL, __FUNCTION__);
        return RC_PARAMENT_ERROR;
    }
	
	current_filemapping_mgr = *filemapping_mgr_ptr;

	memset(current_filemapping_mgr.filemapping[part_index].file_name,0,sizeof(current_filemapping_mgr.filemapping[part_index].file_name));
	memcpy(current_filemapping_mgr.filemapping[part_index].file_name,name,(strlen(name)>sizeof(current_filemapping_mgr.filemapping[part_index].file_name)?sizeof(current_filemapping_mgr.filemapping[part_index].file_name):strlen(name)));

    //SET_NSTRING(current_filemapping_mgr.filemapping[part_index].file_name,name,sizeof(current_filemapping_mgr.filemapping[part_index].file_name));

    current_filemapping_mgr.filemapping[part_index].file_size = file_size;
    current_filemapping_mgr.filemapping[part_index].creat_time = 0;
    //current_filemapping_mgr.filemapping[part_index].part_index = part_index;
    current_filemapping_mgr.filemapping[part_index].file_type = type;
    //current_filemapping_mgr.filemapping[part_index].part_type = PART_TYPE_FIXED;
	//current_filemapping_mgr.filemapping[part_index].startup_flag &= ~AMS_PARTITION_FREE;
    //current_filemapping_mgr.filemapping[part_index].startup_flag |= AMS_PARTITION_INACTIVE;
	//current_filemapping_mgr.filemapping[part_index].part_checksum =CheckSum((unsigned long)&(current_filemapping_mgr.filemapping[part_index].file_name),sizeof(AMS_FILEMAPPING_T) - sizeof(unsigned long));
    
	AMS_PART_write_filemapping2flash(&current_filemapping_mgr);

    return RC_OK;
}

unsigned long 
AMS_PART_get_info_by_name(unsigned char *name, 
                          unsigned long *addr, 
                          unsigned long *size,
                          AMS_PARTITION_TYPE_T *type)
{
    unsigned long i;
    FlashRegion_T region;
    unsigned long part_num;

    if (name == NULL || addr == NULL || size == NULL || type == NULL)
    {
        AMS_PRINTK("<%d>%s:invalid params\n", AMS_PART_LOG_LEVEL, __FUNCTION__);
        return RC_PARAMENT_ERROR;
    }
    
    AMS_PART_get_number(&part_num);
    for (i = 0; i < part_num; i++)
    {
        if (AMS_PART_get_region(i, &region) == 0)
        { 
            if (strcmp((char *)region.region_name, (char *)name) == 0)
            {
                *addr = region.region_base;
                *size = region.region_size;
                *type = region.region_type;
                return RC_OK;
            }
        }
    }
  
    return RC_ERROR;
}

unsigned long 
AMS_PART_get_info_by_filename(const char *filename, 
                              unsigned long *addr, 
                              unsigned long *size,
                              unsigned long *part_idx,
                              AMS_PARTITION_DEVICE_TYPE_T *device_type)
{
    unsigned long i;
    FlashRegion_T region;
    AMS_FILEMAPPING_T filemapping;
    unsigned long part_num;

    if (filename == NULL || addr == NULL || size == NULL 
        || part_idx == NULL || device_type == NULL)
    {
        AMS_PRINTK("<%d>%s:invalid params\n", AMS_PART_LOG_LEVEL, __FUNCTION__);
        return RC_PARAMENT_ERROR;
    }
    
    AMS_PART_get_number(&part_num);
    for (i = 0; i < part_num; i++)
    {
        if (AMS_PART_get_filemapping(i, &filemapping) == 0)
        {
            if (strcmp((char *)filemapping.file_name, (char *)filename) == 0)
            {
                if (AMS_PART_get_region(i, &region) == 0)
                {
                    *addr = region.region_base;
                    *size = region.region_size;
                    *part_idx = i;
                    if (AMS_PART_get_device_type((char *)region.device_name, device_type) != 0)
                        return RC_ERROR;
                    return RC_OK;
                }
            }
        }
    }
    return RC_ERROR;
}

/*-------------------------------------------------------------------------
 * FUNCTION NAME - AMS_PART_get_partition_filename
 * ------------------------------------------------------------------------
 * PURPOSE  :   This funtion will get partition name by index
 * INPUT      :   index = the index of the partition
 * OUTPUT   :   part_name - the name associated the partition
 * RETURN    :   -1/0
 * NOTES     :   None
 * ------------------------------------------------------------------------
 */	
unsigned long 
AMS_PART_get_partition_filename(unsigned long index, 
    unsigned char *file_name, unsigned long *file_length)
{
    if (file_name == NULL || file_length == NULL)
    {
        AMS_PRINTK("<%d>%s:invalid params\n", AMS_PART_LOG_LEVEL, __FUNCTION__);
        return RC_PARAMENT_ERROR;
    }
    if (index < 0 || index > MAX_PARTITION_NUMBER)
    {
        AMS_PRINTK("<%d>%s:invalid partition index\n", AMS_PART_LOG_LEVEL, __FUNCTION__);
        return RC_PARAMENT_ERROR;
    }

    AMS_PART_get_filemapping_mgr_ptr(&filemapping_mgr_ptr);
#if 0
    if(filemapping_mgr_ptr->filemapping[index].part_checksum != 
        CheckSum((unsigned long)&(filemapping_mgr_ptr->filemapping[index].file_name), 
        sizeof(AMS_FILEMAPPING_T) - sizeof(unsigned long)))
    {
        return RC_ERROR;
    }
#endif
    strcpy((char *)file_name,GET_STRING(filemapping_mgr_ptr->filemapping[index].file_name));
    *file_length = filemapping_mgr_ptr->filemapping[index].file_size;

    return RC_OK;
}

/*-------------------------------------------------------------------------
 * FUNCTION NAME - AMS_PART_get_region
 * ------------------------------------------------------------------------
 * PURPOSE  :   This funtion will get partition region from partition table
 * INPUT      :   index = the index of partition
 * OUTPUT   :   region = the region of the given partition index. 
 * RETURN    :   The execute of this function
 * NOTES     :   None
 * ------------------------------------------------------------------------
 */
unsigned long 
AMS_PART_get_region(unsigned long index, FlashRegion_T *region)
{
    if (region == NULL)
    {
        AMS_PRINTK("<%d>%s:invalid params\n", AMS_PART_LOG_LEVEL, __FUNCTION__);
        return RC_PARAMENT_ERROR;
    }

    if (index < 0 || index >= MAX_PARTITION_NUMBER)
        return RC_PART_NUMBER_EXCEED;

    if (AMS_PART_check_partition() != 0)
        return RC_PART_CHECK_ERROR;

    if (pt_ptr->flash_region[index].region_type == TYPE_NOT_USED)
        return RC_PART_DO_NOT_EXIST;

    if (pt_ptr->flash_region[index].region_checksum != 
           CheckSum((unsigned long)&pt_ptr->flash_region[index].region_name, 
                sizeof(FlashRegion_T) - sizeof(unsigned long)))
        return RC_PART_DO_NOT_EXIST;

    *region = pt_ptr->flash_region[index];

    return RC_OK;
}

unsigned long
AMS_PART_get_boot_part(AMS_PARTITION_TYPE_T type, unsigned long *active_part_idx)
{
    int i;
	unsigned long part_num = 0;
	AMS_FILEMAPPING_T filemapping;

    if (active_part_idx == NULL)
    {
        AMS_PRINTK("<%d>%s:invalid params\n", AMS_PART_LOG_LEVEL, __FUNCTION__);
        return RC_PARAMENT_ERROR;
    }

#ifndef ACCTON_PATCH    
	if (type < TYPE_NOT_USED || type > TYPE_MAX)
	{
		AMS_PRINTK("<%d>%s:invalid part type\n", AMS_PART_LOG_LEVEL, __FUNCTION__);
		return RC_ERROR;
	}
#endif

    AMS_PART_get_filemapping_mgr_ptr(&filemapping_mgr_ptr);
	AMS_PART_get_number(&part_num);
	for (i = 0; i < part_num; i++)
	{
       	if (pt_ptr->flash_region[i].region_type != type)
            continue;
        
       	if (filemapping_mgr_ptr->filemapping[i].startup_flag & AMS_PARTITION_ACTIVE)
       	{
       		*active_part_idx = i;
			break;
       	}
	}

	if (AMS_PART_get_filemapping(*active_part_idx, &filemapping) !=  0)
	{
        AMS_PRINTK("<%d>%s:get part error by index\n", AMS_PART_LOG_LEVEL, __FUNCTION__);
		return RC_ERROR;
	}

	if (filemapping.startup_flag & AMS_PARTITION_FREE)
	{
		AMS_PRINTK("<%d>%s:boot part is free\n", AMS_PART_LOG_LEVEL, __FUNCTION__);
		return RC_ERROR;
	}
	
	if (!(filemapping.startup_flag & AMS_PARTITION_ACTIVE))
	{
		AMS_PRINTK("<%d>%s:boot part is not active\n", AMS_PART_LOG_LEVEL, __FUNCTION__);
		return RC_ERROR;
	}
	return RC_OK;
}

/*-------------------------------------------------------------------------
 * FUNCTION NAME - AMS_PART_get_inactive_part
 * ------------------------------------------------------------------------
 * PURPOSE  :   Get index of the inactive partition with specific type.
 * INPUT    :   type = partition type
 * OUTPUT   :   part_idx = index of the partition. 
 * RETURN   :   
 *              RC_OK = Success
 *              <others> = Fail
 * NOTES    :   None
 * ------------------------------------------------------------------------
 */
int AMS_PART_get_inactive_part(AMS_PARTITION_TYPE_T type, unsigned long *part_idx)
{
    int i;
    AMS_FILEMAPPING_MGR_T *fmt;

    if (part_idx == NULL)
    {
        printf("%s:invalid params\n", __FUNCTION__);
        return RC_PARAMENT_ERROR;
    }

    AMS_PART_get_filemapping_mgr_ptr(&fmt);

    for (i = 0; i < MAX_PARTITION_NUMBER; i++)
    {
        if (pt_ptr->flash_region[i].region_type != type)
            continue;

        if (fmt->filemapping[i].startup_flag & AMS_PARTITION_FREE)
            continue;

        if (fmt->filemapping[i].startup_flag & AMS_PARTITION_INACTIVE)
        {
            *part_idx = i;
            return RC_OK;
        }
    }

    return RC_ERROR;
}

/* Don't do real protect for all sectors.
 * SPI Flash write protection on per sector is not supported,
 * and some partitions need to be read/write in runtime.
 */
#if 0
static unsigned long
AMS_PART_write_flash(unsigned long dest_addr, unsigned char *src, 
     unsigned long erase_length, unsigned long write_length, unsigned char protect)
{

    unsigned long err_code=RC_ERROR;

    debug("%s(): [dest_addr] 0x%X [src] 0x%X [erase_length] %ld [write_length] %ld [protect] %d\n",
          __FUNCTION__, dest_addr, src, erase_length, write_length, protect);
   
 
    if (src == NULL)
    {
        AMS_PRINTK("<%d>%s:invalid params\n", AMS_PART_LOG_LEVEL, __FUNCTION__);
        return RC_PARAMENT_ERROR;
    }

    
    /* map flash address to physical address and do flash chip select */ 
    AMS_FLASH_MAPPING(dest_addr);

    err_code = RC_OK;    
    /* flag protect means that the partition is protected, we should protect off it when write to it */
    if (protect > 0 )
    {
        if (flash_sect_protect(0, dest_addr, dest_addr + erase_length - 1))
        {
            AMS_PRINTK("<%d>%s:un-protech flash error\n", AMS_PART_LOG_LEVEL, __FUNCTION__);
            err_code = RC_ERROR;
        }
    }

    if (flash_sect_erase(dest_addr, dest_addr + erase_length - 1))
    {
        AMS_PRINTK("<%d>%s:erase flash error\n", AMS_PART_LOG_LEVEL, __FUNCTION__);
        err_code = RC_ERROR;
    }

    if (flash_write((char *)src, dest_addr, write_length))
    {
        AMS_PRINTK("<%d>%s:write flash error\n", AMS_PART_LOG_LEVEL, __FUNCTION__);
        err_code = RC_ERROR;
    }

    if (protect == 1)
    {
        if (flash_sect_protect(1, dest_addr, dest_addr + erase_length - 1)){
            AMS_PRINTK("<%d>%s:protech flash error\n", AMS_PART_LOG_LEVEL, __FUNCTION__);
            err_code = RC_ERROR;
        }
    }
    
    AMS_FLASH_DEFAULT();    /* select on first flash as normal operation */ 

    return err_code;

    return RC_ERROR;
}
#endif
#if 0
unsigned long
AMS_PART_write_part2flash(AMS_PARTITION_TABLE_T *partition_table_ptr)
{
    AMS_PARTITION_DEVICE_TYPE_T device_type;
    FlashRegion_T   region;
    unsigned long region_mode;    
    unsigned char p = 1;  // do real protect

    /* get flash device type */
    device_type = TYPE_DEVICE_SPI;  /* default flash type */
    if(AMS_PART_get_region(PARTITION_MANAGEMENT_INDEX, &region) == RC_OK){
        AMS_PART_get_device_type((char *)region.device_name, &device_type);
    }
    debug("device type = %ld\n", device_type);

    /* Don't do real protect for all sectors.
     * SPI Flash write protection on per sector is not supported,
     * and some partitions need to be read/write in runtime.
     */
    region_mode=AMS_PART_get_region_mode(&region);
    if ((region_mode == PART_MODE_RW) || (device_type == TYPE_DEVICE_SPI))
        p = 2;  //Don't do real protect
    

    AMS_PART_write_flash(PARTITION_MANAGEMENT_BASE_ADDR, (unsigned char *)partition_table_ptr, 
        CFG_FLASH_SECTOR_SIZE, sizeof(AMS_PARTITION_TABLE_T), p);

    /* when the partition table is changed, we get new filemapping region address */
    AMS_PART_get_filemapping_mgr_ptr(&filemapping_mgr_ptr);

    return RC_OK;
}
#endif
#if 0
unsigned long 
AMS_PART_write_filemapping2flash(AMS_FILEMAPPING_MGR_T *filemapping_mgr)
{
    AMS_PARTITION_DEVICE_TYPE_T device_type;
    FlashRegion_T region;
    unsigned long region_mode;
    unsigned char p = 1;  // do real protect

    if (AMS_PART_get_filemapping_region(&region) == RC_OK)
    {
        AMS_PART_get_device_type((char *)region.device_name, &device_type);
        debug("device type = %ld\n", device_type);

        /* Don't do real protect for all sectors.
         * SPI Flash write protection on per sector is not supported,
         * and some partitions need to be read/write in runtime.
         */    
        region_mode=AMS_PART_get_region_mode(&region);
        if ((region_mode == PART_MODE_RW) || (device_type == TYPE_DEVICE_SPI))
            p = 2;  //Don't do real protect
            
        return AMS_PART_write_flash(region.region_base, (unsigned char *)filemapping_mgr, 
                CFG_FLASH_SECTOR_SIZE, sizeof(AMS_FILEMAPPING_MGR_T), p);
    }
    else
        return RC_ERROR;
}
#endif
unsigned long AMS_PART_write_to_device(AMS_PARTITION_TYPE_T type,
                                  const char *name,
                                  unsigned long addr, 
                                  unsigned long length,
                                  unsigned long part_index)
{
    unsigned long ret = RC_ERROR;
#if 0
    AMS_PARTITION_DEVICE_TYPE_T device_type;
    unsigned long start_addr = 0, size = 0;
    unsigned long region_mode;

    FlashRegion_T   region;
    unsigned char p = 1;  // do real protect

    debug("\n%s(): [type] %d [name] %s [addr] 0x%X [length] %ld\n",__FUNCTION__, type, name, addr, length);

    if (type <= TYPE_NOT_USED || type >= TYPE_MAX){
        printf("[AMS] Invalid type of partation. type = %d\n", type);
		return RC_ERROR;		
    }

    if (AMS_PART_get_info_by_type(type, &start_addr, &size, part_index, &device_type) != RC_OK){
	   	printf("[AMS] Fail to find partition. type = %d, index = %ld\n", type, part_index);
		return RC_ERROR;
        
    }else{            
        debug("flash partation info:  start_addr=0x%X, size=%ld, device_type=%d\n", start_addr, size, device_type);
        
        if(AMS_PART_get_region(part_index, &region) != RC_OK){
	   	    printf("[AMS] Fail to find region. part_index = %d\n", part_index);
		    return RC_ERROR;            
        }
        region_mode=AMS_PART_get_region_mode(&region);
        
        switch (device_type){
            case TYPE_DEVICE_NOR:
                if(region_mode == PART_MODE_RW)
                    p = 2;  //Don't do real protect

                ret = AMS_PART_write_flash(start_addr, (unsigned char *)addr, size, length, p);
                break;
                
            case TYPE_DEVICE_SPI: 
                /* SPI Flash write protection on per sector is not supported */
                ret = AMS_PART_write_flash(start_addr, (unsigned char *)addr, size, length, 2);
                break;

            #ifdef CONFIG_COMPACT_FLASH
            case TYPE_DEVICE_CFCARD:
                if (length > size){
                    AMS_PRINTK("<%d>%s:too large length\n", AMS_PART_LOG_LEVEL, __FUNCTION__);
                    ret = RC_ERROR;
                    goto END;
                }
			   
                /* compact flash's write/read operation should be sector sized */
                start_addr /= CF_STANDARD_SECTOR_SIZE;
                /* 
                 * the format operation will cost a lot of time, write data to cf card directly
                 */
                size = length / CF_STANDARD_SECTOR_SIZE + 1;
                if (AMS_CFLIB_write_data(start_addr, size, (unsigned char*)(addr), FALSE)!= 0){
                    AMS_PRINTK("<%d>%s:write cfcard error\n", AMS_PART_LOG_LEVEL, __FUNCTION__);
                    ret = RC_ERROR;
                    goto END;
                }
                break;
            #endif
            
            default:
                printf("[AMS] Unknow device type. device_type = %d\n", __FUNCTION__, device_type);
        }        
    }

    if(ret == RC_OK){
        debug("\nset partition info: \n");
        if (AMS_PART_set_part_info_by_index(part_index, name, length, type) != RC_OK){
            printf("[AMS] Fail to set partition info.\n");
            ret = RC_ERROR;
        }
    }
    
END:
#endif
    return ret;
}

unsigned long
AMS_PART_read_from_device(AMS_PARTITION_TYPE_T type,
                                     const char *name,
                                     unsigned long length, 
                                     unsigned long *buf_addr)
{
    unsigned long ret = RC_ERROR;
#if 0
    AMS_PARTITION_DEVICE_TYPE_T device_type;
    unsigned long start_addr = 0;
    unsigned long size = 0;
    unsigned long partition_index = 0;

    if (type <  TYPE_NOT_USED || type >= TYPE_MAX)
    {
        AMS_PRINTK("<%d>%s:invalid params\n", AMS_PART_LOG_LEVEL, __FUNCTION__);
		return RC_ERROR;
    }

    debug("AMS_PART_read_from_device type %d, name %s, length %ld, *buf_addr 0x%08x\n", type, name, length, *buf_addr);
    if (AMS_PART_get_info_by_filename((char *)name, &start_addr, &size, &partition_index,&device_type) == 0)
    {
        switch (device_type)
		{
            case TYPE_DEVICE_NOR:
                *buf_addr = start_addr;
                break;

            case TYPE_DEVICE_SPI:
                if(!*buf_addr){
                    printf("[AMS] Invalid buffer address ! Null pointer !\n");
                    return RC_ERROR;
                }

                /* map flash address to physical address and do flash chip select */ 
                AMS_FLASH_MAPPING(start_addr);                
                //memmove ((char *)*buf_addr, (char *)start_addr, length);  
                memmovel (*buf_addr, start_addr, length); //EPR: ES3526MA-HPoE-7LF-LN-00016, Speed booting process.
                AMS_FLASH_DEFAULT();    /* select on first flash as normal operation */                 
                break;

            case TYPE_DEVICE_NAND:
                break;

#ifdef CONFIG_COMPACT_FLASH
            case TYPE_DEVICE_CFCARD:
                start_addr /= CF_STANDARD_SECTOR_SIZE;
                length = length / CF_STANDARD_SECTOR_SIZE + 1;
                AMS_CFLIB_read_data(start_addr, length, (unsigned char *)*buf_addr);
                break;
#endif
            default:
               AMS_PRINTK("<%d>%s:invalid device type\n", AMS_PART_LOG_LEVEL, __FUNCTION__);
               break;
        }
    }
    else
    {
        AMS_PRINTK("<%d>%s:failed to find linux part\n", AMS_PART_LOG_LEVEL, __FUNCTION__);
        return RC_ERROR;
    }
    return RC_OK;
#endif
    return ret;
}
#if 0
unsigned long AMS_PART_get_region_mode(FlashRegion_T* region)
{
    FlashRegionExt_T* regionExt;

    regionExt = (FlashRegionExt_T*)region;
    debug("%s (%s)\n", regionExt->region_name, (regionExt->region_mode ==PART_MODE_RO)? "RO":"RW");

    return regionExt->region_mode;
}
#endif
/* ------------------------------------------------------------------------
 * FUNCTION NAME: AMS_PART_get_non_startup_filename
 * PURPOSE  : get the inactive file of a given type.
 * INPUT    : file_type   -- the type of file
 * OUTPUT   : 
 *            file_name   -- the file name 
 *            file_length -- the size of file.
 * RETURN   : 
 *            RC_OK    = Success
 *            RC_ERROR = Fail
 * NOTES    : None
 * ------------------------------------------------------------------------
 */
int AMS_PART_get_non_startup_filename(AMS_PARTITION_TYPE_T file_type, unsigned char *file_name, unsigned long *file_length)
{
    unsigned long part_idx;
    int result = RC_ERROR;

    if (AMS_PART_get_inactive_part(file_type, &part_idx) == RC_OK)
    {
        if (AMS_PART_get_partition_filename(part_idx, file_name, file_length) == RC_OK)
        {
            result= RC_OK;
        }
        else
        {
            debug("%s(): get part filename error ! type = %ld, part_index = %ld\n", __FUNCTION__, file_type, part_idx);
        }

    }
    else
    {
        debug("%s(): get inactive part error !  type =%ld\n", __FUNCTION__, file_type);
    }

    return result;
}

/* ------------------------------------------------------------------------
 * FUNCTION NAME: AMS_PART_get_startup_filename
 * PURPOSE  : get the startup file of a given type.
 * INPUT    : 
 *            file_type   -- the type of file
 * OUTPUT   :
 *            file_name   -- the file name 
 *            file_length -- the size of file.
 * RETURN   :
 *            RC_OK    = Success
 *            RC_ERROR = Fail
 * NOTES    : None
 * ------------------------------------------------------------------------
 */
int AMS_PART_get_startup_filename(AMS_PARTITION_TYPE_T file_type, unsigned char *file_name, unsigned long *file_length)
{
    unsigned long part_idx = 0;
    int result = RC_ERROR;

    if (AMS_PART_get_boot_part(file_type, &part_idx) == RC_OK)
    {
        if (AMS_PART_get_partition_filename(part_idx, file_name, file_length) == RC_OK)
        {
            result= RC_OK;
        }
        else
        {
            debug("%s(): get part filename error ! type = %ld, part_index = %ld\n", __FUNCTION__, file_type, part_idx);
        }
    }
    else
    {
        debug("%s(): get boot part error !  type =%ld\n", __FUNCTION__, file_type);
    }

    return result;
}


/* ------------------------------------------------------------------------
 * FUNCTION NAME: AMS_PART_recover_runtime_to_default
 * PURPOSE  :   Recover runtime file information and
 *              write the default file name, size.
 *              Bootable runtime image will be loaded at CFG_LOAD_ADDR if it is
 *              recovered successfully.
 * INPUT    :   None
 * OUTPUT   :   None
 * RETURN   :   
 *              RC_OK    = Success
 *              RC_ERROR = Fail
 *
 * NOTES    :   This function shall check the runtime file in flash first.
 *              If runtime's checksum is correct, then using default file name.
 *              If there is environment "runtimename", get file name from "runtimename".
 *              Otherwise, runtime file name is the partition name with extension ".bix".
 *
 *              If runtime's checksum is not correct, then check next runtime file.
 * ------------------------------------------------------------------------
 */
int AMS_PART_recover_runtime_to_default(void)
{
    int i, result=RC_ERROR;
    image_header_t  header;
    char file_name[PART_FILE_NAMELEN];
    char *env_runtimefile_name;
    unsigned long checksum, file_size, runtime_base_addr;

    for (i = 0; i < MAX_PARTITION_NUMBER; i++)
    {
        debug("%s:region %d: %ld\n", __FUNCTION__, i, pt_ptr->flash_region[i].region_type);

        if (pt_ptr->flash_region[i].region_type != TYPE_RUNTIME)
            continue;

        debug("%s:Runtime partition index = %d\n", __FUNCTION__,  i);

        runtime_base_addr=pt_ptr->flash_region[i].region_base;
        AMS_FLASH_MAPPING(runtime_base_addr);
        udelay(10000);
        memmove(&header, (char *)runtime_base_addr, sizeof(image_header_t));
        AMS_FLASH_DEFAULT();

        /* vaildate magic
         */
        if (header.ih_magic != IH_MAGIC)
        {
            debug("%s:Bad runtime file(image header magic byte error) on part idx %d(read val=%u)!\n", __FUNCTION__, i, header.ih_magic);
            continue;
        }

        /* validate image header crc
         */
        checksum = ntohl(header.ih_hcrc);
        header.ih_hcrc = 0;
        if(crc32(0, (uchar*)&header, sizeof(image_header_t)) != checksum)
        {
            debug("%s:Bad image header checksum\n", __FUNCTION__);
            continue;
        }

        /* Runtime file name
         */
        env_runtimefile_name = getenv("runtimename");
        if(env_runtimefile_name)
            strcpy(file_name, env_runtimefile_name);
        else
            sprintf(file_name, "%s.bix", pt_ptr->flash_region[i].region_name); 

        file_size=sizeof(image_header_t)+ntohl(header.ih_size);
        debug("%s:Set to default: [part idx] %d [file name] %s, [file size] %ld\n", __FUNCTION__, i, file_name, file_size);

        if (AMS_PART_set_part_info_by_index(i, file_name, file_size, TYPE_RUNTIME) == RC_OK)
        {
            unsigned long boot_addr = CFG_LOAD_ADDR;

            if(AMS_PART_read_from_device(TYPE_RUNTIME, file_name, file_size, &boot_addr)==RC_OK)
            {
                /* validate image data crc
                 */
                if(crc32(0, (uchar*)boot_addr + sizeof(image_header_t), file_size - sizeof(image_header_t))!=ntohl(header.ih_dcrc))
                {
                    /* delete invalid runtime file
                     */
                    debug("%s:Bad image data checksum. Delete file '%s'\n", __FUNCTION__, file_name);
                    AMS_PART_delete_runtime_file(file_name);
                    continue;
                }
            }
            else
            {
                debug("%s:Read part index %d error. Delete file '%s'\n", __FUNCTION__, i, file_name);
                /* delete invalid runtime file
                 */
                AMS_PART_delete_runtime_file(file_name);
            }

            /* recover successfully
             */
            result=RC_OK;
            break;
        }
        else
        {
            debug("%s: Fail to set part idx %d as default\n", __FUNCTION__, i);
            continue;
        }

    }

    return result;
}

/* ------------------------------------------------------------------------
 * FUNCTION NAME: AMS_PART_validate_and_load_boot_runtime
 * PURPOSE  :   Validate the startup runtime image(the runtime image set as active)
 *              The runtime image will be loaded to CFG_LOAD_ADDR after returning
 *              from this function if there is no error.
 * INPUT    :   None
 * OUTPUT   :   None
 * RETURN   :   
 *              RC_OK    = Success
 *              RC_ERROR = Fail
 *
 * NOTES    : 1.This function will validate the startup runtime image through
 *              image header checksum and image data checksum. The runtime
 *              image will be loaded to CFG_LOAD_ADDR if there is no error.
 *            2.Invalid runtime image will be deleted from file mapping table.
 * ------------------------------------------------------------------------
 */
int AMS_PART_validate_and_load_boot_runtime(void)
{
    unsigned long active_part_idx, checksum, boot_addr = CFG_LOAD_ADDR;
    unsigned long file_size;
    image_header_t  header;
    char file_name[SYS_ADPT_FILE_SYSTEM_NAME_LEN+1];

    if(AMS_PART_get_boot_part(TYPE_RUNTIME, &active_part_idx)!=RC_OK)
        return RC_ERROR;

    if(AMS_PART_get_partition_filename(active_part_idx, (unsigned char*)file_name, &file_size)!=RC_OK)
        return RC_ERROR;

    if(AMS_PART_read_from_device(TYPE_RUNTIME, file_name, file_size, &boot_addr)==RC_OK)
    {
        memmove(&header, (char *)boot_addr, sizeof(image_header_t));
        /* vaildate magic
         */
        if (header.ih_magic != IH_MAGIC)
        {
            debug("%s:Bad runtime file(image header magic byte error)!, file_name=%s\n", __FUNCTION__, file_name);
            return RC_ERROR;
        }

        /* validate image header crc
         */
        checksum = ntohl(header.ih_hcrc);
        header.ih_hcrc = 0;
        if(crc32(0, (uchar*)&header, sizeof(image_header_t)) != checksum)
        {
            debug("%s:Bad image header checksum, file_name=%s\n", __FUNCTION__, file_name);
            return RC_ERROR;
        }

        /* validate image data crc
         */
        if(crc32(0, (uchar*)boot_addr + sizeof(image_header_t), file_size-sizeof(image_header_t))!=ntohl(header.ih_dcrc))
        {
            debug("%s:Bad image data checksum, file_name=%s\n", __FUNCTION__, file_name);
            return RC_ERROR;
        }
    }
    else
        return RC_ERROR;

    return RC_OK;
}

/* ------------------------------------------------------------------------
 * FUNCTION NAME: AMS_PART_delete_runtime_file
 * PURPOSE:	delete runtime file in flash device.
 * INPUT:	file name
 * OUTPUT:	None
 * RETURN:	
 *              RC_OK    = Success
 *              RC_ERROR = Fail
 * NOTES:
 * ------------------------------------------------------------------------
 */
int AMS_PART_delete_runtime_file(char *file_name)
{
#if 0
    AMS_PARTITION_DEVICE_TYPE_T device_type;
    AMS_PARTITION_TYPE_T type = TYPE_RUNTIME;
    unsigned long start_addr = 0;
    unsigned long size = 0;
    unsigned long partition_index = 0;
    unsigned long ret;
    char *argv[3];

    if(AMS_PART_get_info_by_filename((char *)file_name, &start_addr, &size, &partition_index,&device_type) != RC_OK)
    {
        debug("%s:Cannot find file - %s", __FUNCTION__, file_name);
        return RC_ERROR;
    }

    debug("%s:[partition_index]%d, [start_addr]0x%X, [size]%ld\n", __FUNCTION__, partition_index, start_addr, size);

    if(AMS_PART_check_active(partition_index, type) == RC_OK)
    {
        ret = AMS_CMD_PART_set_startup_flag(partition_index, AMS_PARTITION_INACTIVE);
        
        if (ret == RC_PART_NUMBER_EXCEED || ret == RC_PART_DO_NOT_EXIST)
        {
            debug("%s:Invalid partition index: %d", __FUNCTION__, partition_index);
            return RC_ERROR;
        }
    }

    argv[0] = &file_name[0];
    argv[1] = &file_name[0];
    if(do_delete(NULL, 0, 2, argv)==0)
        return RC_OK;

    debug("%s:delete file error", __FUNCTION__);
#endif
    return RC_ERROR;
}

/* ------------------------------------------------------------------------
 * FUNCTION NAME: AMS_PART_prepare_runtime_image
 * PURPOSE:	Check file partition table and select a bootable runtime image.
 *          If a bootable runtime image is not available, this function will
 *          try to recover a usable runtime image in flash through the
 *          validation of checksum.
 * INPUT:	None
 * OUTPUT:	boot_runtime_filename -- the filename of the runtime that is
 *                                   being used to boot up.
 *                                   The size of boot_runtime_filename must be
 *                                   greater or equal to PART_FILE_NAMELEN.
 * RETURN:	RC_OK    = Success
 *          RC_ERROR = Fail (No runtime image availabe for boot)
 * NOTES: 1.If RC_OK is returned, the bootable runtime image will be loaded
 *          at CFG_LOAD_ADDR.
 *        2.Invalid runtime will be deleted from file mapping table.
 *        3.The startup runtime could be changed if the original startup
 *          runtime image is invalid.
 * ------------------------------------------------------------------------
 */
int AMS_PART_prepare_runtime_image(char* boot_runtime_filename)
{
#if 0
    int img_runtime = FALSE;    /* The default runtime image is ready or not */
    unsigned long length;
    char          *local_args[2];

    /************************************************************************
     * Check default image file and get file name from file system.
     */
    if (AMS_PART_get_startup_filename(TYPE_RUNTIME, (unsigned char*)boot_runtime_filename, &length) == RC_OK)
    {
        if (AMS_PART_validate_and_load_boot_runtime()==RC_OK)
        {
            img_runtime = TRUE;
            debug("The default runtime image : %s, file size = %ld\r\n", boot_runtime_filename, length);
        }
        else
        {
            /* delete invalid runtime image
             */
            printf("Delete invalid runtime: %s\r\n", boot_runtime_filename);
            AMS_PART_delete_runtime_file((char*)boot_runtime_filename);

            /* default startup runtime image is invalid, try to use another
             * non-startup runtime image
             */
            printf("The default runtime image is invalid( %s, file size = %ld)\r\n", boot_runtime_filename, length);
            if (AMS_PART_get_non_startup_filename(TYPE_RUNTIME, (unsigned char*)boot_runtime_filename, &length) == RC_OK)
            {
                local_args[0] = (char *)boot_runtime_filename;
                local_args[1] = (char *)boot_runtime_filename;
                do_startup_file(NULL, 0, 2,local_args);
                /* validate the new startup runtime image
                 */
                if (AMS_PART_validate_and_load_boot_runtime()==RC_OK)
                {
                    img_runtime = TRUE;
                    printf("Set startup runtime image : %s, file size = %ld\r\n", boot_runtime_filename, length);
                }
                else
                {
                    printf("Non startup runtime image is invalid either. filename:%s, file size = %ld\r\n", boot_runtime_filename, length);
                    AMS_PART_delete_runtime_file((char*)boot_runtime_filename);
                }
            }
            else
            {
                printf("Non startup runtime image doesn not exist.\r\n");
                /* try our best to recover the possible usable runtimage on flash
                 */
                if (AMS_PART_recover_runtime_to_default()==RC_OK)
                {
                    if (AMS_PART_get_non_startup_filename(TYPE_RUNTIME, boot_runtime_filename, &length) == RC_OK)
                    {
                        img_runtime = TRUE;
                        printf("Runtime image is recovered successfully. (startup runtime to %s)\r\n", boot_runtime_filename);
                        local_args[0] = (char *)boot_runtime_filename;
                        local_args[1] = (char *)boot_runtime_filename;
                        do_startup_file(NULL, 0, 2,local_args); // set this file as startup type.
                        printf("The recovered runtime image : %s, file size = %ld\r\n", boot_runtime_filename, length);
                    }
                }
                else
                {
                    printf("Recover runtime image failed\r\n");
                }
                
            }
        }
    }
    else if (AMS_PART_get_non_startup_filename(TYPE_RUNTIME, (unsigned char*)boot_runtime_filename, &length) == RC_OK)
    {
        /* read the non-startup runtime image file from file system and start it.
         */
        printf("No startup runtime is set. Set startup runtime to %s\r\n", boot_runtime_filename);
        local_args[0] = (char *)boot_runtime_filename;
        local_args[1] = (char *)boot_runtime_filename;
        do_startup_file(NULL, 0, 2,local_args); // set this file as startup type.
        if (AMS_PART_validate_and_load_boot_runtime()==RC_OK)
        {
            img_runtime = TRUE;
            debug("The default runtime image : %s, file size = %ld\r\n", boot_runtime_filename, length);
        }
        else
        {
            printf("Delete invalid runtime image %s .\r\n", boot_runtime_filename);
            AMS_PART_delete_runtime_file((char*)boot_runtime_filename);

            /* try to get another non-startup runtime image file from file system
             */
            if(AMS_PART_get_non_startup_filename(TYPE_RUNTIME, (unsigned char*)boot_runtime_filename, &length) == RC_OK)
            {
                /* try to set as start-up and boot
                 */
                local_args[0] = (char *)boot_runtime_filename;
                local_args[1] = (char *)boot_runtime_filename;
                do_startup_file(NULL, 0, 2,local_args); // set this file as startup type.
                if (AMS_PART_validate_and_load_boot_runtime()==RC_OK)
                {
                    img_runtime = TRUE;
                    printf("The default runtime image : %s, file size = %ld\r\n", boot_runtime_filename, length);
                }
                else
                {
                    printf("Delete invalid runtime image %s .\r\n", boot_runtime_filename);
                    AMS_PART_delete_runtime_file((char*)boot_runtime_filename);
                }
            }
            else
            {
                /* try to recover one runtime
                 */
                debug("Try to recover a bootable runtime\r\n");
                if (AMS_PART_recover_runtime_to_default()==RC_OK)
                {
                    if (AMS_PART_get_non_startup_filename(TYPE_RUNTIME, (unsigned char*)boot_runtime_filename, &length) == RC_OK)
                    {
                        img_runtime = TRUE;
                        printf("runtime image is recovered sucessfully and will be set as startup runtime.(filename:%s)\r\n", boot_runtime_filename);
                        local_args[0] = (char *)boot_runtime_filename;
                        local_args[1] = (char *)boot_runtime_filename;
                        do_startup_file(NULL, 0, 2,local_args); // set this file as startup type.
                        debug("The default runtime image : %s, file size = %ld\r\n", boot_runtime_filename, length);
                    }
                }
            }
        }
    }
    else
    {
        /* No runtime file in file system, try to recover runtime.
         */
        printf("No runtime file in file system, try to recover.\r\n");
        if (AMS_PART_recover_runtime_to_default()==RC_OK)
        {
            if (AMS_PART_get_non_startup_filename(TYPE_RUNTIME, (unsigned char*)boot_runtime_filename, &length) == RC_OK)
            {
                img_runtime = TRUE;
                printf("runtime image is recovered sucessfully and will be set as startup runtime.(filename:%s)\r\n", boot_runtime_filename);
                local_args[0] = (char *)boot_runtime_filename;
                local_args[1] = (char *)boot_runtime_filename;
                do_startup_file(NULL, 0, 2,local_args); // set this file as startup type.
                debug("The default runtime image : %s, file size = %ld\r\n", boot_runtime_filename, length);
            }
        }
    }

    return (img_runtime==TRUE)? RC_OK : RC_ERROR;
#endif
    return RC_ERROR;
}



